-- Water Vending System Database
-- MySQL 5.7+

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";

CREATE DATABASE IF NOT EXISTS `water_vending` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE `water_vending`;

-- Admin users
CREATE TABLE `admin_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(50) NOT NULL,
  `email` varchar(100) NOT NULL,
  `password` varchar(255) NOT NULL,
  `full_name` varchar(100) NOT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `username` (`username`),
  UNIQUE KEY `email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Clients (water companies)
CREATE TABLE `clients` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `client_code` varchar(20) NOT NULL,
  `company_name` varchar(100) NOT NULL,
  `email` varchar(100) NOT NULL,
  `phone` varchar(20) NOT NULL,
  `address` text,
  `stron_company` varchar(100) NOT NULL,
  `stron_username` varchar(100) NOT NULL,
  `stron_password` varchar(255) NOT NULL,
  `mpesa_paybill` varchar(20) NOT NULL,
  `mpesa_consumer_key` varchar(500),
  `mpesa_consumer_secret` varchar(500),
  `mpesa_passkey` varchar(500),
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `client_code` (`client_code`),
  UNIQUE KEY `mpesa_paybill` (`mpesa_paybill`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Client users (login to client portal)
CREATE TABLE `client_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `client_id` int(11) NOT NULL,
  `username` varchar(50) NOT NULL,
  `email` varchar(100) NOT NULL,
  `password` varchar(255) NOT NULL,
  `full_name` varchar(100) NOT NULL,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `username` (`username`),
  KEY `client_id` (`client_id`),
  FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Meters
CREATE TABLE `meters` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `client_id` int(11) NOT NULL,
  `meter_id` varchar(50) NOT NULL,
  `meter_type` varchar(50) DEFAULT 'WATER',
  `location` varchar(200),
  `is_active` tinyint(1) DEFAULT 1,
  `is_assigned` tinyint(1) DEFAULT 0,
  `total_vended` decimal(15,2) DEFAULT 0,
  `last_vend_date` datetime,
  `last_credit_query` datetime,
  `last_credit_amount` decimal(15,2) DEFAULT 0,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `meter_id` (`meter_id`),
  KEY `client_id` (`client_id`),
  FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Customers
CREATE TABLE `customers` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `client_id` int(11) NOT NULL,
  `customer_id` varchar(50) NOT NULL,
  `first_name` varchar(50) NOT NULL,
  `last_name` varchar(50) NOT NULL,
  `phone` varchar(20) NOT NULL,
  `email` varchar(100),
  `id_number` varchar(50),
  `address` text,
  `is_active` tinyint(1) DEFAULT 1,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `customer_id` (`customer_id`),
  KEY `client_id` (`client_id`),
  KEY `phone` (`phone`),
  FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Meter assignments (one meter = one customer at a time)
CREATE TABLE `meter_assignments` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `meter_id` int(11) NOT NULL,
  `customer_id` int(11) NOT NULL,
  `assigned_at` datetime NOT NULL,
  `assigned_by` int(11),
  `is_active` tinyint(1) DEFAULT 1,
  `unassigned_at` datetime,
  `notes` text,
  PRIMARY KEY (`id`),
  KEY `meter_id` (`meter_id`),
  KEY `customer_id` (`customer_id`),
  KEY `is_active` (`is_active`),
  FOREIGN KEY (`meter_id`) REFERENCES `meters`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`customer_id`) REFERENCES `customers`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Payments from M-Pesa
CREATE TABLE `payments` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `client_id` int(11) NOT NULL,
  `customer_id` int(11),
  `meter_id` int(11),
  `mpesa_tx_id` varchar(50) NOT NULL,
  `amount` decimal(10,2) NOT NULL,
  `paybill` varchar(20) NOT NULL,
  `account_number` varchar(50) NOT NULL,
  `phone` varchar(20) NOT NULL,
  `first_name` varchar(50),
  `tx_date` datetime NOT NULL,
  `status` enum('PENDING','VERIFIED','PROCESSED','FAILED') DEFAULT 'PENDING',
  `error_msg` text,
  `raw_data` text,
  `processed_at` datetime,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `mpesa_tx_id` (`mpesa_tx_id`),
  KEY `client_id` (`client_id`),
  KEY `status` (`status`),
  KEY `account_number` (`account_number`),
  FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Tokens
CREATE TABLE `tokens` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `payment_id` int(11),
  `client_id` int(11) NOT NULL,
  `customer_id` int(11) NOT NULL,
  `meter_id` int(11) NOT NULL,
  `token_value` varchar(255) NOT NULL,
  `token_type` enum('VENDING','CLEAR_CREDIT','CLEAR_TAMPER') DEFAULT 'VENDING',
  `amount` decimal(10,2),
  `units` decimal(10,2),
  `is_vend_by_unit` tinyint(1) DEFAULT 0,
  `is_manual` tinyint(1) DEFAULT 0,
  `issued_by` int(11),
  `status` enum('CREATED','DELIVERED','FAILED') DEFAULT 'CREATED',
  `delivered_at` datetime,
  `sms_to` varchar(20),
  `sms_response` text,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `payment_id` (`payment_id`),
  KEY `client_id` (`client_id`),
  KEY `customer_id` (`customer_id`),
  KEY `meter_id` (`meter_id`),
  FOREIGN KEY (`client_id`) REFERENCES `clients`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`customer_id`) REFERENCES `customers`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`meter_id`) REFERENCES `meters`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Vending requests (for retry logic)
CREATE TABLE `vending_requests` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `payment_id` int(11),
  `token_id` int(11),
  `idempotency_key` varchar(100) NOT NULL,
  `request_data` text,
  `response_data` text,
  `status` enum('PENDING','SUCCESS','FAILED') DEFAULT 'PENDING',
  `error_msg` text,
  `attempt_count` int(11) DEFAULT 0,
  `next_retry_at` datetime,
  `sent_at` datetime,
  `received_at` datetime,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `idempotency_key` (`idempotency_key`),
  KEY `payment_id` (`payment_id`),
  KEY `status` (`status`),
  KEY `next_retry_at` (`next_retry_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- System logs
CREATE TABLE `system_logs` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `level` enum('INFO','WARNING','ERROR','CRITICAL') DEFAULT 'INFO',
  `module` varchar(50) NOT NULL,
  `message` text NOT NULL,
  `context` text,
  `user_id` int(11),
  `ip` varchar(45),
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `level` (`level`),
  KEY `created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default admin (password: admin123)
INSERT INTO `admin_users` (`username`, `email`, `password`, `full_name`) 
VALUES ('admin', 'admin@system.local', '$2y$10$Y.Dc75rnXxOvwQp6XuUGXeP/kfXQa0zYD75NWhh1gHWubJMDaTAKO', 'System Administrator');

INSERT INTO `client_users` (`client_id`, `username`, `email`, `password`, `full_name`, `is_active`) 
VALUES (1, 'trinity', 'trinity@gmail.com', '$2y$10$Y.Dc75rnXxOvwQp6XuUGXeP/kfXQa0zYD75NWhh1gHWubJMDaTAKO', 'Trinity Admin', 1);

-- Analytics views
CREATE VIEW `v_daily_revenue` AS
SELECT 
    DATE(p.tx_date) as date,
    c.id as client_id,
    c.company_name,
    COUNT(p.id) as tx_count,
    SUM(p.amount) as total_amount,
    COUNT(CASE WHEN p.status = 'PROCESSED' THEN 1 END) as success_count
FROM payments p
JOIN clients c ON p.client_id = c.id
GROUP BY DATE(p.tx_date), c.id;

CREATE VIEW `v_customer_stats` AS
SELECT 
    cu.id,
    cu.customer_id,
    CONCAT(cu.first_name, ' ', cu.last_name) as customer_name,
    cu.phone,
    c.company_name,
    COUNT(t.id) as total_tokens,
    SUM(t.amount) as total_spent,
    SUM(t.units) as total_units,
    MAX(t.created_at) as last_vend
FROM customers cu
JOIN clients c ON cu.client_id = c.id
LEFT JOIN tokens t ON cu.id = t.customer_id
GROUP BY cu.id;

CREATE VIEW `v_meter_stats` AS
SELECT 
    m.id,
    m.meter_id,
    m.location,
    c.company_name,
    cu.customer_id,
    CONCAT(cu.first_name, ' ', cu.last_name) as customer_name,
    COUNT(t.id) as total_vends,
    SUM(t.amount) as total_revenue,
    MAX(t.created_at) as last_vend
FROM meters m
JOIN clients c ON m.client_id = c.id
LEFT JOIN meter_assignments ma ON m.id = ma.meter_id AND ma.is_active = 1
LEFT JOIN customers cu ON ma.customer_id = cu.id
LEFT JOIN tokens t ON m.id = t.meter_id
GROUP BY m.id;

