<?php
require_once __DIR__ . '/StronpowerService.php';
require_once __DIR__ . '/SMSService.php';
require_once __DIR__ . '/DB.php';
class VendingService {
    
    private static function calcNextRetry($attemptCount) {
        $minutes = min(pow(2, max($attemptCount - 1, 0)), 60);
        return date('Y-m-d H:i:s', strtotime("+$minutes minutes"));
    }
    
    public static function vendForPayment($paymentId, $idempotencyKey) {
        $db = DB::conn();
        
        // Check existing
        $stmt = $db->prepare("
            SELECT vr.*, t.token_value 
            FROM vending_requests vr
            LEFT JOIN tokens t ON vr.token_id = t.id
            WHERE vr.idempotency_key = ?
        ");
        $stmt->execute([$idempotencyKey]);
        $existing = $stmt->fetch();
        
        if ($existing && $existing['status'] === 'SUCCESS' && $existing['token_value']) {
            return [true, $existing['token_id'], null];
        }
        
        // Get payment details
        $stmt = $db->prepare("
            SELECT p.*, c.id as cid, c.stron_company, c.stron_username, c.stron_password,
                   m.meter_id, m.id as mid,
                   cu.customer_id, cu.id as cuid, cu.phone
            FROM payments p
            JOIN clients c ON p.client_id = c.id
            LEFT JOIN meters m ON p.meter_id = m.id
            LEFT JOIN customers cu ON p.customer_id = cu.id
            WHERE p.id = ?
        ");
        $stmt->execute([$paymentId]);
        $payment = $stmt->fetch();
        
        if (!$payment) {
            return [false, null, 'Payment not found'];
        }
        
        if (!$payment['mid']) {
            return [false, null, 'Meter not found'];
        }
        
        if (!$payment['cuid']) {
            return [false, null, 'Customer not assigned'];
        }
        
        $db->beginTransaction();
        
        try {
            // Create or update vending request
            if ($existing) {
                $vrId = $existing['id'];
                $attemptCount = $existing['attempt_count'] + 1;
                
                $stmt = $db->prepare("
                    UPDATE vending_requests 
                    SET attempt_count = ?, sent_at = NOW(), status = 'PENDING', error_msg = NULL
                    WHERE id = ?
                ");
                $stmt->execute([$attemptCount, $vrId]);
            } else {
                $requestData = json_encode([
                    'meter_id' => $payment['meter_id'],
                    'customer_id' => $payment['customer_id'],
                    'amount' => $payment['amount'],
                    'is_vend_by_unit' => false
                ]);
                
                $stmt = $db->prepare("
                    INSERT INTO vending_requests 
                    (payment_id, idempotency_key, request_data, status, attempt_count, sent_at)
                    VALUES (?, ?, ?, 'PENDING', 1, NOW())
                ");
                $stmt->execute([$paymentId, $idempotencyKey, $requestData]);
                $vrId = $db->lastInsertId();
                $attemptCount = 1;
            }
            
            // Call Stronpower
            $credentials = [
                'company_name' => $payment['stron_company'],
                'username' => $payment['stron_username'],
                'password' => $payment['stron_password']
            ];
            
            $stron = new StronpowerService();
            list($ok, $vendData, $error) = $stron->vendingMeter(
                $credentials,
                $payment['meter_id'],
                $payment['amount'],
                false,
                $payment['customer_id']
            );
            
            if (!$ok || !isset($vendData['token'])) {
                // Failed - schedule retry
                $nextRetry = self::calcNextRetry($attemptCount);
                
                $stmt = $db->prepare("
                    UPDATE vending_requests 
                    SET status = 'FAILED', error_msg = ?, received_at = NOW(), 
                        next_retry_at = ?, response_data = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $error ?: 'Vending failed',
                    $nextRetry,
                    json_encode(['error' => $error, 'raw' => $vendData]),
                    $vrId
                ]);
                
                $db->commit();
                return [false, null, $error ?: 'Vending failed'];
            }
            
            // Success - create token
            $tokenValue = $vendData['token'];
            $units = $vendData['units'] ?? null;
            
            $stmt = $db->prepare("
                INSERT INTO tokens 
                (payment_id, client_id, customer_id, meter_id, token_value, token_type,
                 amount, units, is_vend_by_unit, is_manual, status, created_at)
                VALUES (?, ?, ?, ?, ?, 'VENDING', ?, ?, 0, 0, 'CREATED', NOW())
            ");
            $stmt->execute([
                $paymentId,
                $payment['cid'],
                $payment['cuid'],
                $payment['mid'],
                $tokenValue,
                $payment['amount'],
                $units
            ]);
            $tokenId = $db->lastInsertId();
            
            // Update vending request
            $stmt = $db->prepare("
                UPDATE vending_requests 
                SET status = 'SUCCESS', token_id = ?, received_at = NOW(),
                    response_data = ?, error_msg = NULL, next_retry_at = NULL
                WHERE id = ?
            ");
            $stmt->execute([
                $tokenId,
                json_encode($vendData),
                $vrId
            ]);
            
            // Update payment
            $stmt = $db->prepare("
                UPDATE payments 
                SET status = 'PROCESSED', processed_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$paymentId]);
            
            // Update meter
            $stmt = $db->prepare("
                UPDATE meters 
                SET last_vend_date = NOW(), total_vended = total_vended + ?
                WHERE id = ?
            ");
            $stmt->execute([$payment['amount'], $payment['mid']]);
            
            $db->commit();
            
            // Send SMS
            if ($payment['phone']) {
                $sms = new SMSService();
                list($smsOk, $smsResp) = $sms->sendToken(
                    $payment['phone'],
                    $payment['meter_id'],
                    $tokenValue,
                    $payment['amount'],
                    $units ?? 0
                );
                
                if ($smsOk) {
                    $stmt = $db->prepare("
                        UPDATE tokens 
                        SET status = 'DELIVERED', delivered_at = NOW(),
                            sms_to = ?, sms_response = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([
                        $payment['phone'],
                        $smsResp,
                        $tokenId
                    ]);
                }
            }
            
            return [true, $tokenId, null];
            
        } catch (Exception $e) {
            $db->rollBack();
            return [false, null, $e->getMessage()];
        }
    }
    
    public static function vendManually($meterId, $customerId, $amount, $isVendByUnit, $issuedBy) {
        $db = DB::conn();
        $db->beginTransaction();
        
        try {
            // Get details
            $stmt = $db->prepare("
                SELECT m.id as mid, m.meter_id, m.client_id,
                       c.stron_company, c.stron_username, c.stron_password,
                       cu.id as cuid, cu.customer_id, cu.phone
                FROM meters m
                JOIN clients c ON m.client_id = c.id
                JOIN customers cu ON cu.id = ? AND cu.client_id = c.id
                WHERE m.id = ?
            ");
            $stmt->execute([$customerId, $meterId]);
            $data = $stmt->fetch();
            
            if (!$data) {
                throw new Exception('Meter or customer not found');
            }
            
            // Call Stronpower
            $credentials = [
                'company_name' => $data['stron_company'],
                'username' => $data['stron_username'],
                'password' => $data['stron_password']
            ];
            
            $stron = new StronpowerService();
            list($ok, $vendData, $error) = $stron->vendingMeter(
                $credentials,
                $data['meter_id'],
                $amount,
                $isVendByUnit,
                $data['customer_id']
            );
            
            if (!$ok || !isset($vendData['token'])) {
                throw new Exception($error ?: 'Vending failed');
            }
            
            // Create token
            $tokenValue = $vendData['token'];
            $units = $vendData['units'] ?? null;
            
            $stmt = $db->prepare("
                INSERT INTO tokens 
                (client_id, customer_id, meter_id, token_value, token_type,
                 amount, units, is_vend_by_unit, is_manual, issued_by, status, created_at)
                VALUES (?, ?, ?, ?, 'VENDING', ?, ?, ?, 1, ?, 'CREATED', NOW())
            ");
            $stmt->execute([
                $data['client_id'],
                $customerId,
                $meterId,
                $tokenValue,
                $amount,
                $units,
                $isVendByUnit ? 1 : 0,
                $issuedBy
            ]);
            $tokenId = $db->lastInsertId();
            
            // Update meter
            $stmt = $db->prepare("
                UPDATE meters 
                SET last_vend_date = NOW(), total_vended = total_vended + ?
                WHERE id = ?
            ");
            $stmt->execute([$amount, $meterId]);
            
            $db->commit();
            
            // Send SMS
            if ($data['phone']) {
                $sms = new SMSService();
                list($smsOk, $smsResp) = $sms->sendToken(
                    $data['phone'],
                    $data['meter_id'],
                    $tokenValue,
                    $amount,
                    $units ?? 0
                );
                
                if ($smsOk) {
                    $stmt = $db->prepare("
                        UPDATE tokens 
                        SET status = 'DELIVERED', delivered_at = NOW(),
                            sms_to = ?, sms_response = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([
                        $data['phone'],
                        $smsResp,
                        $tokenId
                    ]);
                }
            }
            
            return [true, $tokenId, $tokenValue];
            
        } catch (Exception $e) {
            $db->rollBack();
            return [false, null, $e->getMessage()];
        }
    }
    
    public static function retryFailed() {
        $db = DB::conn();
        
        $stmt = $db->prepare("
            SELECT vr.*, p.id as payment_id
            FROM vending_requests vr
            JOIN payments p ON vr.payment_id = p.id
            WHERE vr.status = 'FAILED' 
            AND vr.next_retry_at IS NOT NULL
            AND vr.next_retry_at <= NOW()
            AND vr.attempt_count < 5
            LIMIT 10
        ");
        $stmt->execute();
        $requests = $stmt->fetchAll();
        
        $results = [];
        foreach ($requests as $req) {
            list($ok, $tokenId, $error) = self::vendForPayment(
                $req['payment_id'],
                $req['idempotency_key']
            );
            
            $results[] = [
                'vr_id' => $req['id'],
                'payment_id' => $req['payment_id'],
                'success' => $ok,
                'error' => $error
            ];
        }
        
        return $results;
    }
}
