<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$success = $error = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        try {
            $password = password_hash($_POST['password'], PASSWORD_BCRYPT);
            $stmt = $db->prepare("
                INSERT INTO client_users 
                (client_id, username, email, password, full_name, is_active)
                VALUES (?, ?, ?, ?, ?, 1)
            ");
            $stmt->execute([
                $_POST['client_id'],
                $_POST['username'],
                $_POST['email'],
                $password,
                $_POST['full_name']
            ]);
            $success = 'Client user created successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'update') {
        try {
            if (!empty($_POST['password'])) {
                $password = password_hash($_POST['password'], PASSWORD_BCRYPT);
                $stmt = $db->prepare("
                    UPDATE client_users 
                    SET username = ?, email = ?, password = ?, full_name = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $_POST['username'],
                    $_POST['email'],
                    $password,
                    $_POST['full_name'],
                    $_POST['id']
                ]);
            } else {
                $stmt = $db->prepare("
                    UPDATE client_users 
                    SET username = ?, email = ?, full_name = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $_POST['username'],
                    $_POST['email'],
                    $_POST['full_name'],
                    $_POST['id']
                ]);
            }
            $success = 'Client user updated successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'toggle') {
        $stmt = $db->prepare("UPDATE client_users SET is_active = NOT is_active WHERE id = ?");
        $stmt->execute([$_POST['id']]);
        $success = 'User status updated';
    } elseif ($action === 'delete') {
        try {
            $stmt = $db->prepare("DELETE FROM client_users WHERE id = ?");
            $stmt->execute([$_POST['id']]);
            $success = 'Client user deleted successfully';
        } catch (PDOException $e) {
            $error = 'Error: Cannot delete user - ' . $e->getMessage();
        }
    }
}

// Get all client users
$users = $db->query("
    SELECT cu.*, c.company_name, c.client_code
    FROM client_users cu
    JOIN clients c ON cu.client_id = c.id
    ORDER BY cu.created_at DESC
")->fetchAll();

// Get clients for dropdown
$clients = $db->query("SELECT id, client_code, company_name FROM clients WHERE is_active = 1 ORDER BY company_name")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Client User Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Client User Management</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openModal()" class="btn btn-orange">
                        <i class="fas fa-plus"></i> Add Client User
                    </button>
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>All Client Users (<?= count($users) ?>)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Client</th>
                                <th>Username</th>
                                <th>Full Name</th>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $u): ?>
                                <tr>
                                    <td><strong>#<?= $u['id'] ?></strong></td>
                                    <td>
                                        <?= htmlspecialchars($u['company_name']) ?><br>
                                        <small><?= htmlspecialchars($u['client_code']) ?></small>
                                    </td>
                                    <td><strong><?= htmlspecialchars($u['username']) ?></strong></td>
                                    <td><?= htmlspecialchars($u['full_name']) ?></td>
                                    <td><?= htmlspecialchars($u['email']) ?></td>
                                    <td>
                                        <?php if ($u['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge badge-danger">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= date('Y-m-d', strtotime($u['created_at'])) ?></td>
                                    <td>
                                        <button onclick="editUser(<?= htmlspecialchars(json_encode($u)) ?>)" class="btn btn-sm btn-primary">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <form method="POST" style="display:inline">
                                            <input type="hidden" name="action" value="toggle">
                                            <input type="hidden" name="id" value="<?= $u['id'] ?>">
                                            <button type="submit" class="btn btn-sm" style="background:#f78510;color:white" title="Toggle Status">
                                                <i class="fas fa-power-off"></i>
                                            </button>
                                        </form>
                                        <form method="POST" style="display:inline" onsubmit="return confirm('Delete this user? This cannot be undone.')">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $u['id'] ?>">
                                            <button type="submit" class="btn btn-sm" style="background:#dc3545;color:white">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Modal -->
    <div id="userModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitle">Add Client User</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST" id="userForm">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="userId">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Client *</label>
                        <select name="client_id" id="clientId" required>
                            <option value="">Select Client</option>
                            <?php foreach ($clients as $c): ?>
                                <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['company_name']) ?> (<?= htmlspecialchars($c['client_code']) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Username *</label>
                            <input type="text" name="username" id="username" required>
                        </div>
                        <div class="form-group">
                            <label>Full Name *</label>
                            <input type="text" name="full_name" id="fullName" required>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Email *</label>
                            <input type="email" name="email" id="email" required>
                        </div>
                        <div class="form-group">
                            <label>Password <span id="passwordNote">*</span></label>
                            <input type="password" name="password" id="password">
                            <small style="color:#666" id="passwordHelp">Minimum 6 characters</small>
                        </div>
                    </div>
                    
                    <div class="alert" style="background:#d1ecf1;border-left-color:#17a2b8;color:#0c5460">
                        <strong>Note:</strong> This user will be able to login to the client portal and manage customers, meters, and tokens for their assigned client.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange" id="submitBtn">Create User</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openModal() {
        document.getElementById('modalTitle').textContent = 'Add Client User';
        document.getElementById('formAction').value = 'create';
        document.getElementById('submitBtn').textContent = 'Create User';
        document.getElementById('userForm').reset();
        document.getElementById('userId').value = '';
        document.getElementById('password').required = true;
        document.getElementById('passwordNote').textContent = '*';
        document.getElementById('passwordHelp').textContent = 'Minimum 6 characters';
        document.getElementById('userModal').classList.add('active');
    }
    
    function editUser(user) {
        document.getElementById('modalTitle').textContent = 'Edit Client User';
        document.getElementById('formAction').value = 'update';
        document.getElementById('submitBtn').textContent = 'Update User';
        document.getElementById('userId').value = user.id;
        document.getElementById('clientId').value = user.client_id;
        document.getElementById('username').value = user.username;
        document.getElementById('fullName').value = user.full_name;
        document.getElementById('email').value = user.email;
        document.getElementById('password').value = '';
        document.getElementById('password').required = false;
        document.getElementById('passwordNote').textContent = '(leave blank to keep current)';
        document.getElementById('passwordHelp').textContent = 'Leave blank to keep current password';
        document.getElementById('userModal').classList.add('active');
    }
    
    function closeModal() {
        document.getElementById('userModal').classList.remove('active');
    }
    
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    
    // Password validation
    document.getElementById('userForm').addEventListener('submit', function(e) {
        const password = document.getElementById('password').value;
        const action = document.getElementById('formAction').value;
        
        if (action === 'create' && password.length < 6) {
            e.preventDefault();
            alert('Password must be at least 6 characters long');
            return false;
        }
        
        if (action === 'update' && password && password.length < 6) {
            e.preventDefault();
            alert('Password must be at least 6 characters long');
            return false;
        }
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>