<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$success = $error = '';

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        try {
            $stmt = $db->prepare("
                INSERT INTO clients 
                (client_code, company_name, email, phone, address, 
                 stron_company, stron_username, stron_password,
                 mpesa_paybill, mpesa_consumer_key, mpesa_consumer_secret, mpesa_passkey)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['client_code'],
                $_POST['company_name'],
                $_POST['email'],
                $_POST['phone'],
                $_POST['address'],
                $_POST['stron_company'],
                $_POST['stron_username'],
                $_POST['stron_password'],
                $_POST['mpesa_paybill'],
                $_POST['mpesa_consumer_key'],
                $_POST['mpesa_consumer_secret'],
                $_POST['mpesa_passkey']
            ]);
            $success = 'Client created successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'toggle') {
        $id = $_POST['id'];
        $stmt = $db->prepare("UPDATE clients SET is_active = NOT is_active WHERE id = ?");
        $stmt->execute([$id]);
        $success = 'Client status updated';
    }
}

// Get all clients
$clients = $db->query("
    SELECT c.*, 
           COUNT(DISTINCT m.id) as meter_count,
           COUNT(DISTINCT cu.id) as customer_count,
           COALESCE(SUM(p.amount), 0) as total_revenue
    FROM clients c
    LEFT JOIN meters m ON c.id = m.client_id
    LEFT JOIN customers cu ON c.id = cu.client_id
    LEFT JOIN payments p ON c.id = p.client_id AND p.status = 'PROCESSED'
    GROUP BY c.id
    ORDER BY c.created_at DESC
")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Client Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Client Management</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openModal()" class="btn btn-orange">
                        <i class="fas fa-plus"></i> Add Client
                    </button>
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>All Clients</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Company</th>
                                <th>Contact</th>
                                <th>Paybill</th>
                                <th>Meters</th>
                                <th>Customers</th>
                                <th>Revenue</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($clients as $c): ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($c['client_code']) ?></strong></td>
                                    <td><?= htmlspecialchars($c['company_name']) ?></td>
                                    <td>
                                        <?= htmlspecialchars($c['email']) ?><br>
                                        <small><?= htmlspecialchars($c['phone']) ?></small>
                                    </td>
                                    <td><code><?= htmlspecialchars($c['mpesa_paybill']) ?></code></td>
                                    <td><?= $c['meter_count'] ?></td>
                                    <td><?= $c['customer_count'] ?></td>
                                    <td>KES <?= number_format($c['total_revenue'], 2) ?></td>
                                    <td>
                                        <?php if ($c['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge badge-danger">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <form method="POST" style="display:inline">
                                            <input type="hidden" name="action" value="toggle">
                                            <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                            <button type="submit" class="btn btn-sm" onclick="return confirm('Toggle status?')">
                                                <i class="fas fa-power-off"></i>
                                            </button>
                                        </form>
                                        <button onclick="viewClient(<?= htmlspecialchars(json_encode($c)) ?>)" class="btn btn-sm btn-primary">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Client Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New Client</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Client Code *</label>
                            <input type="text" name="client_code" required>
                        </div>
                        <div class="form-group">
                            <label>Company Name *</label>
                            <input type="text" name="company_name" required>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Email *</label>
                            <input type="email" name="email" required>
                        </div>
                        <div class="form-group">
                            <label>Phone *</label>
                            <input type="text" name="phone" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Address</label>
                        <textarea name="address" rows="2"></textarea>
                    </div>
                    
                    <h4 style="margin:1rem 0 0.5rem;color:var(--blue)">Stronpower Credentials</h4>
                    <div class="grid grid-cols-3 gap-4">
                        <div class="form-group">
                            <label>Company Name *</label>
                            <input type="text" name="stron_company" required>
                        </div>
                        <div class="form-group">
                            <label>Username *</label>
                            <input type="text" name="stron_username" required>
                        </div>
                        <div class="form-group">
                            <label>Password *</label>
                            <input type="text" name="stron_password" required>
                        </div>
                    </div>
                    
                    <h4 style="margin:1rem 0 0.5rem;color:var(--blue)">M-Pesa Configuration</h4>
                    <div class="form-group">
                        <label>Paybill Number *</label>
                        <input type="text" name="mpesa_paybill" required>
                    </div>
                    <div class="grid grid-cols-3 gap-4">
                        <div class="form-group">
                            <label>Consumer Key</label>
                            <input type="text" name="mpesa_consumer_key">
                        </div>
                        <div class="form-group">
                            <label>Consumer Secret</label>
                            <input type="text" name="mpesa_consumer_secret">
                        </div>
                        <div class="form-group">
                            <label>Passkey</label>
                            <input type="text" name="mpesa_passkey">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Create Client</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openModal() {
        document.getElementById('addModal').classList.add('active');
    }
    function closeModal() {
        document.getElementById('addModal').classList.remove('active');
    }
    function viewClient(client) {
        alert('Client Details:\n\n' + 
              'Company: ' + client.company_name + '\n' +
              'Stronpower: ' + client.stron_company + '\n' +
              'Paybill: ' + client.mpesa_paybill);
    }
    
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>