<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$success = $error = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'query_credit') {
        require_once '../../includes/StronpowerService.php';
        
        $meterId = $_POST['meter_id'];
        $meterIdValue = $_POST['meter_id_value'];
        
        // Get client credentials
        $credStmt = $db->prepare("
            SELECT c.stron_company, c.stron_username, c.stron_password
            FROM meters m
            JOIN clients c ON m.client_id = c.id
            WHERE m.id = ?
        ");
        $credStmt->execute([$meterId]);
        $creds = $credStmt->fetch();
        
        if ($creds) {
            $credentials = [
                'company_name' => $creds['stron_company'],
                'username' => $creds['stron_username'],
                'password' => $creds['stron_password']
            ];
            
            $stron = new StronpowerService();
            list($ok, $data, $err) = $stron->queryMeterCredit($credentials, $meterIdValue);
            
            if ($ok) {
                // Extract credit from response
                $credit = $data['Credit'] ?? $data['credit'] ?? $data['TotalCredit'] ?? $data['total_credit'] ?? 0;
                
                // Update meter record
                $updateStmt = $db->prepare("
                    UPDATE meters 
                    SET last_credit_query = NOW(), 
                        last_credit_amount = ?
                    WHERE id = ?
                ");
                $updateStmt->execute([$credit, $meterId]);
                
                $success = "Meter Credit: KES " . number_format($credit, 2);
            } else {
                $error = "Failed to query meter credit: " . ($err ?: 'Unknown error');
            }
        } else {
            $error = "Meter or client not found";
        }
    } elseif ($action === 'create') {
        try {
            $stmt = $db->prepare("
                INSERT INTO meters (client_id, meter_id, meter_type, location)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['client_id'],
                $_POST['meter_id'],
                $_POST['meter_type'],
                $_POST['location']
            ]);
            $success = 'Meter added successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'toggle') {
        $stmt = $db->prepare("UPDATE meters SET is_active = NOT is_active WHERE id = ?");
        $stmt->execute([$_POST['id']]);
        $success = 'Meter status updated';
    }
}

// Get all meters with client info
$meters = $db->query("
    SELECT m.*, c.company_name, c.client_code,
           cu.customer_id, CONCAT(cu.first_name, ' ', cu.last_name) as customer_name,
           COUNT(t.id) as token_count,
           COALESCE(SUM(t.amount), 0) as total_revenue
    FROM meters m
    JOIN clients c ON m.client_id = c.id
    LEFT JOIN meter_assignments ma ON m.id = ma.meter_id AND ma.is_active = 1
    LEFT JOIN customers cu ON ma.customer_id = cu.id
    LEFT JOIN tokens t ON m.id = t.meter_id
    GROUP BY m.id
    ORDER BY m.created_at DESC
")->fetchAll();

// Get clients for dropdown
$clients = $db->query("SELECT id, client_code, company_name FROM clients WHERE is_active = 1 ORDER BY company_name")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Meter Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Meter Management</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openModal()" class="btn btn-orange">
                        <i class="fas fa-plus"></i> Add Meter
                    </button>
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>All Meters (<?= count($meters) ?>)</h2>
                        <div class="flex gap-2">
                            <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                            <select id="filterClient" style="padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                                <option value="">All Clients</option>
                                <?php foreach ($clients as $c): ?>
                                    <option value="<?= $c['client_code'] ?>"><?= htmlspecialchars($c['company_name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Meter ID</th>
                                <th>Client</th>
                                <th>Location</th>
                                <th>Type</th>
                                <th>Customer</th>
                                <th>Tokens</th>
                                <th>Revenue</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($meters as $m): ?>
                                <tr data-client="<?= htmlspecialchars($m['client_code']) ?>">
                                    <td><strong><?= htmlspecialchars($m['meter_id']) ?></strong></td>
                                    <td><?= htmlspecialchars($m['company_name']) ?></td>
                                    <td><?= htmlspecialchars($m['location'] ?: 'N/A') ?></td>
                                    <td><?= htmlspecialchars($m['meter_type']) ?></td>
                                    <td>
                                        <?php if ($m['customer_name']): ?>
                                            <?= htmlspecialchars($m['customer_name']) ?><br>
                                            <small><?= htmlspecialchars($m['customer_id']) ?></small>
                                        <?php else: ?>
                                            <span class="badge badge-warning">Unassigned</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= $m['token_count'] ?></td>
                                    <td>KES <?= number_format($m['total_revenue'], 2) ?></td>
                                    <td>
                                        <?php if ($m['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge badge-danger">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button onclick="queryCredit(<?= $m['id'] ?>, '<?= htmlspecialchars($m['meter_id']) ?>')" 
                                                class="btn btn-sm btn-primary" title="Query Credit Balance">
                                            <i class="fas fa-wallet"></i>
                                        </button>
                                        <form method="POST" style="display:inline">
                                            <input type="hidden" name="action" value="toggle">
                                            <input type="hidden" name="id" value="<?= $m['id'] ?>">
                                            <button type="submit" class="btn btn-sm" onclick="return confirm('Toggle status?')">
                                                <i class="fas fa-power-off"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Meter Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New Meter</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="create">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Client *</label>
                        <select name="client_id" required>
                            <option value="">Select Client</option>
                            <?php foreach ($clients as $c): ?>
                                <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['company_name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Meter ID *</label>
                        <input type="text" name="meter_id" placeholder="e.g. 58100000007" required>
                        <small style="color:#666">Must match Stronpower meter ID</small>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div class="form-group">
                            <label>Meter Type</label>
                            <select name="meter_type">
                                <option value="WATER">Water</option>
                                <option value="ENERGY">Energy</option>
                                <option value="GAS">Gas</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Location</label>
                            <input type="text" name="location" placeholder="e.g. Building A, Floor 2">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Add Meter</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openModal() {
        document.getElementById('addModal').classList.add('active');
    }
    function closeModal() {
        document.getElementById('addModal').classList.remove('active');
    }
    
    function queryCredit(meterId, meterIdValue) {
        if (confirm('Query credit balance for meter ' + meterIdValue + '?\n\nThis will make a real-time call to Stronpower API.')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="query_credit">
                <input type="hidden" name="meter_id" value="${meterId}">
                <input type="hidden" name="meter_id_value" value="${meterIdValue}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    
    // Filter by client
    document.getElementById('filterClient').addEventListener('change', function(e) {
        const client = e.target.value;
        document.querySelectorAll('tbody tr').forEach(row => {
            if (!client || row.dataset.client === client) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>