<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$success = $error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'delete') {
        try {
            $stmt = $db->prepare("DELETE FROM payments WHERE id = ?");
            $stmt->execute([$_POST['id']]);
            $success = 'Payment deleted successfully';
        } catch (PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    } elseif ($action === 'update_status') {
        $stmt = $db->prepare("UPDATE payments SET status = ? WHERE id = ?");
        $stmt->execute([$_POST['status'], $_POST['id']]);
        $success = 'Payment status updated';
    }
}

// Filters
$filterClient = $_GET['client'] ?? '';
$filterStatus = $_GET['status'] ?? '';
$filterDate = $_GET['date'] ?? '';

$query = "
    SELECT p.*, 
           c.company_name, c.client_code,
           m.meter_id,
           cu.customer_id, CONCAT(cu.first_name, ' ', cu.last_name) as customer_name,
           t.token_value
    FROM payments p
    JOIN clients c ON p.client_id = c.id
    LEFT JOIN meters m ON p.meter_id = m.id
    LEFT JOIN customers cu ON p.customer_id = cu.id
    LEFT JOIN tokens t ON p.id = t.payment_id
    WHERE 1=1
";

$params = [];
if ($filterClient) {
    $query .= " AND c.id = ?";
    $params[] = $filterClient;
}
if ($filterStatus) {
    $query .= " AND p.status = ?";
    $params[] = $filterStatus;
}
if ($filterDate) {
    $query .= " AND DATE(p.tx_date) = ?";
    $params[] = $filterDate;
}

$query .= " ORDER BY p.tx_date DESC LIMIT 100";

$stmt = $db->prepare($query);
$stmt->execute($params);
$payments = $stmt->fetchAll();

// Get clients for filter
$clients = $db->query("SELECT id, company_name FROM clients ORDER BY company_name")->fetchAll();

// Statistics
$stats = [
    'total' => $db->query("SELECT COUNT(*) as c FROM payments")->fetch()['c'],
    'today' => $db->query("SELECT COUNT(*) as c FROM payments WHERE DATE(tx_date)=CURDATE()")->fetch()['c'],
    'pending' => $db->query("SELECT COUNT(*) as c FROM payments WHERE status='PENDING'")->fetch()['c'],
    'processed' => $db->query("SELECT COUNT(*) as c FROM payments WHERE status='PROCESSED'")->fetch()['c'],
    'failed' => $db->query("SELECT COUNT(*) as c FROM payments WHERE status='FAILED'")->fetch()['c'],
    'today_revenue' => $db->query("SELECT COALESCE(SUM(amount),0) as c FROM payments WHERE DATE(tx_date)=CURDATE() AND status='PROCESSED'")->fetch()['c']
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Payment Management</h1>
                <div class="flex items-center gap-4">
                    <span><?= htmlspecialchars($_SESSION['admin_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <!-- Statistics -->
                <div class="grid grid-cols-4 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Total Payments</h3>
                        <div class="value"><?= $stats['total'] ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Today</h3>
                        <div class="value"><?= $stats['today'] ?></div>
                    </div>
                    <div class="stat-card">
                        <h3>Today Revenue</h3>
                        <div class="value">KES <?= number_format($stats['today_revenue'], 2) ?></div>
                    </div>
                    <div class="stat-card orange">
                        <h3>Pending</h3>
                        <div class="value"><?= $stats['pending'] ?></div>
                    </div>
                </div>
                
                <div class="grid grid-cols-2 gap-4 mb-4">
                    <div class="stat-card">
                        <h3>Processed</h3>
                        <div class="value"><?= $stats['processed'] ?></div>
                    </div>
                    <div class="stat-card" style="border-left-color:#dc3545">
                        <h3>Failed</h3>
                        <div class="value"><?= $stats['failed'] ?></div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="bg-white p-4 rounded shadow mb-4">
                    <form method="GET" class="grid grid-cols-4 gap-4">
                        <div class="form-group">
                            <label>Client</label>
                            <select name="client">
                                <option value="">All Clients</option>
                                <?php foreach ($clients as $c): ?>
                                    <option value="<?= $c['id'] ?>" <?= $filterClient == $c['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($c['company_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Status</label>
                            <select name="status">
                                <option value="">All Status</option>
                                <option value="PENDING" <?= $filterStatus == 'PENDING' ? 'selected' : '' ?>>Pending</option>
                                <option value="VERIFIED" <?= $filterStatus == 'VERIFIED' ? 'selected' : '' ?>>Verified</option>
                                <option value="PROCESSED" <?= $filterStatus == 'PROCESSED' ? 'selected' : '' ?>>Processed</option>
                                <option value="FAILED" <?= $filterStatus == 'FAILED' ? 'selected' : '' ?>>Failed</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Date</label>
                            <input type="date" name="date" value="<?= htmlspecialchars($filterDate) ?>">
                        </div>
                        <div class="form-group" style="display:flex;align-items:flex-end;gap:0.5rem">
                            <button type="submit" class="btn btn-primary">Filter</button>
                            <a href="payments.php" class="btn">Reset</a>
                        </div>
                    </form>
                </div>
                
                <!-- Payments Table -->
                <div class="table-container">
                    <div class="table-header">
                        <h2>Payments (Last 100)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Tx ID</th>
                                <th>Client</th>
                                <th>Meter</th>
                                <th>Customer</th>
                                <th>Phone</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Token</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($payments as $p): ?>
                                <tr>
                                    <td><?= date('Y-m-d H:i', strtotime($p['tx_date'])) ?></td>
                                    <td><code style="font-size:0.7rem"><?= htmlspecialchars($p['mpesa_tx_id']) ?></code></td>
                                    <td><?= htmlspecialchars($p['company_name']) ?></td>
                                    <td><?= htmlspecialchars($p['meter_id'] ?: 'N/A') ?></td>
                                    <td><?= htmlspecialchars($p['customer_name'] ?: 'N/A') ?></td>
                                    <td><?= htmlspecialchars($p['phone']) ?></td>
                                    <td><strong>KES <?= number_format($p['amount'], 2) ?></strong></td>
                                    <td>
                                        <?php
                                        $colors = [
                                            'PROCESSED' => 'badge-success',
                                            'PENDING' => 'badge-warning',
                                            'VERIFIED' => 'badge-info',
                                            'FAILED' => 'badge-danger'
                                        ];
                                        ?>
                                        <span class="badge <?= $colors[$p['status']] ?? 'badge-info' ?>">
                                            <?= $p['status'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($p['token_value']): ?>
                                            <code style="font-size:0.7rem"><?= htmlspecialchars(substr($p['token_value'], 0, 15)) ?>...</code>
                                        <?php else: ?>
                                            <span class="badge badge-warning">No Token</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button onclick="viewPayment(<?= htmlspecialchars(json_encode($p)) ?>)" class="btn btn-sm btn-primary">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button onclick="changeStatus(<?= $p['id'] ?>, '<?= $p['status'] ?>')" class="btn btn-sm" style="background:#f78510;color:white">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <form method="POST" style="display:inline" onsubmit="return confirm('Delete this payment?')">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $p['id'] ?>">
                                            <button type="submit" class="btn btn-sm" style="background:#dc3545;color:white">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- View Payment Modal -->
    <div id="viewModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Payment Details</h3>
                <button onclick="closeViewModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <div class="modal-body" id="paymentDetails">
                <!-- Populated by JS -->
            </div>
            <div class="modal-footer">
                <button onclick="closeViewModal()" class="btn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Change Status Modal -->
    <div id="statusModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Change Payment Status</h3>
                <button onclick="closeStatusModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="update_status">
                <input type="hidden" name="id" id="statusPaymentId">
                <div class="modal-body">
                    <div class="form-group">
                        <label>New Status</label>
                        <select name="status" id="statusSelect" required>
                            <option value="PENDING">Pending</option>
                            <option value="VERIFIED">Verified</option>
                            <option value="PROCESSED">Processed</option>
                            <option value="FAILED">Failed</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeStatusModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Update Status</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    
    function viewPayment(payment) {
        const html = `
            <div style="line-height:1.8">
                <p><strong>Transaction ID:</strong> ${payment.mpesa_tx_id}</p>
                <p><strong>Client:</strong> ${payment.company_name}</p>
                <p><strong>Meter ID:</strong> ${payment.meter_id || 'N/A'}</p>
                <p><strong>Customer:</strong> ${payment.customer_name || 'N/A'}</p>
                <p><strong>Phone:</strong> ${payment.phone}</p>
                <p><strong>Amount:</strong> KES ${parseFloat(payment.amount).toFixed(2)}</p>
                <p><strong>Paybill:</strong> ${payment.paybill}</p>
                <p><strong>Account:</strong> ${payment.account_number}</p>
                <p><strong>Status:</strong> <span class="badge">${payment.status}</span></p>
                <p><strong>Date:</strong> ${payment.tx_date}</p>
                <p><strong>Token:</strong> ${payment.token_value || 'Not generated'}</p>
                ${payment.error_msg ? '<p><strong>Error:</strong> <span style="color:red">' + payment.error_msg + '</span></p>' : ''}
            </div>
        `;
        document.getElementById('paymentDetails').innerHTML = html;
        document.getElementById('viewModal').classList.add('active');
    }
    
    function closeViewModal() {
        document.getElementById('viewModal').classList.remove('active');
    }
    
    function changeStatus(id, currentStatus) {
        document.getElementById('statusPaymentId').value = id;
        document.getElementById('statusSelect').value = currentStatus;
        document.getElementById('statusModal').classList.add('active');
    }
    
    function closeStatusModal() {
        document.getElementById('statusModal').classList.remove('active');
    }
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>