<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';
require_once '../../includes/VendingService.php';
require_once '../../includes/StronpowerService.php';
require_once '../../includes/SMSService.php';

header('Content-Type: application/json');

$raw = file_get_contents('php://input');
$payload = json_decode($raw, true);

file_put_contents('../../logs/mpesa_confirmation_'.date('Ymd').'.log', 
    date('Y-m-d H:i:s')." - ".json_encode($payload)."\n", FILE_APPEND);

// Extract fields
$txId = $payload['TransID'] ?? $payload['TransactionID'] ?? null;
$amount = $payload['TransAmount'] ?? $payload['Amount'] ?? null;
$paybill = $payload['BusinessShortCode'] ?? $payload['ShortCode'] ?? null;

if (!$paybill && isset($payload['ReceiverPartyPublicName'])) {
    $parts = explode(' ', $payload['ReceiverPartyPublicName']);
    $paybill = trim($parts[0]);
}

$meterId = $payload['BillRefNumber'] ?? $payload['AccountReference'] ?? $payload['AccountNumber'] ?? null;
$phone = $payload['MSISDN'] ?? $payload['CustomerMSISDN'] ?? $payload['PhoneNumber'] ?? null;
$firstName = $payload['FirstName'] ?? '';

// Validate required fields
if (!$txId || !$amount || !$paybill || !$meterId || !$phone) {
    echo json_encode(['ResultCode' => 1, 'ResultDesc' => 'Missing required fields']);
    exit;
}

try {
    $db = DB::conn();
    
    // Check duplicate
    $stmt = $db->prepare("SELECT id FROM payments WHERE mpesa_tx_id = ?");
    $stmt->execute([$txId]);
    if ($stmt->fetch()) {
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Already processed']);
        exit;
    }
    
    // Find client by paybill
    $stmt = $db->prepare("
        SELECT id, stron_company, stron_username, stron_password
        FROM clients 
        WHERE mpesa_paybill = ? AND is_active = 1
    ");
    $stmt->execute([$paybill]);
    $client = $stmt->fetch();
    
    if (!$client) {
        // Log failed payment
        $stmt = $db->prepare("
            INSERT INTO payments 
            (client_id, mpesa_tx_id, amount, paybill, account_number, phone, 
             first_name, tx_date, status, error_msg, raw_data)
            VALUES (NULL, ?, ?, ?, ?, ?, ?, NOW(), 'FAILED', 'Client not found', ?)
        ");
        $stmt->execute([$txId, $amount, $paybill, $meterId, $phone, $firstName, json_encode($payload)]);
        
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Client not configured']);
        exit;
    }
    
    // Find meter
    $stmt = $db->prepare("
        SELECT id, meter_id, client_id 
        FROM meters 
        WHERE meter_id = ? AND is_active = 1
    ");
    $stmt->execute([$meterId]);
    $meter = $stmt->fetch();
    
    if (!$meter) {
        $stmt = $db->prepare("
            INSERT INTO payments 
            (client_id, mpesa_tx_id, amount, paybill, account_number, phone, 
             first_name, tx_date, status, error_msg, raw_data)
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), 'FAILED', 'Meter not found', ?)
        ");
        $stmt->execute([$client['id'], $txId, $amount, $paybill, $meterId, $phone, $firstName, json_encode($payload)]);
        
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Meter not found']);
        exit;
    }
    
    // Verify meter belongs to client
    if ($meter['client_id'] != $client['id']) {
        $stmt = $db->prepare("
            INSERT INTO payments 
            (client_id, meter_id, mpesa_tx_id, amount, paybill, account_number, phone, 
             first_name, tx_date, status, error_msg, raw_data)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'FAILED', 'Meter not linked to paybill', ?)
        ");
        $stmt->execute([$client['id'], $meter['id'], $txId, $amount, $paybill, $meterId, $phone, $firstName, json_encode($payload)]);
        
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Meter not linked']);
        exit;
    }
    
    // Get active customer assignment
    $stmt = $db->prepare("
        SELECT customer_id 
        FROM meter_assignments 
        WHERE meter_id = ? AND is_active = 1
        ORDER BY assigned_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$meter['id']]);
    $assignment = $stmt->fetch();
    
    if (!$assignment) {
        $stmt = $db->prepare("
            INSERT INTO payments 
            (client_id, meter_id, mpesa_tx_id, amount, paybill, account_number, phone, 
             first_name, tx_date, status, error_msg, raw_data)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'FAILED', 'Meter not assigned', ?)
        ");
        $stmt->execute([$client['id'], $meter['id'], $txId, $amount, $paybill, $meterId, $phone, $firstName, json_encode($payload)]);
        
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Meter not assigned']);
        exit;
    }
    
    $customerId = $assignment['customer_id'];
    
    // Create payment record
    $stmt = $db->prepare("
        INSERT INTO payments 
        (client_id, customer_id, meter_id, mpesa_tx_id, amount, paybill, 
         account_number, phone, first_name, tx_date, status, raw_data)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'PENDING', ?)
    ");
    $stmt->execute([$client['id'], $customerId, $meter['id'], $txId, $amount, $paybill, $meterId, $phone, $firstName, json_encode($payload)]);
    $paymentId = $db->lastInsertId();
    
    // Attempt vending
    list($success, $tokenId, $error) = VendingService::vendForPayment($paymentId, $txId);
    
    if ($success) {
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Success']);
    } else {
        // Payment accepted, will retry vending
        echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
    }
    
} catch (Exception $e) {
    file_put_contents('../../logs/mpesa_error_'.date('Ymd').'.log', 
        date('Y-m-d H:i:s')." - ".$e->getMessage()."\n", FILE_APPEND);
    
    echo json_encode(['ResultCode' => 1, 'ResultDesc' => 'System error']);
}
