<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';

if (!isset($_SESSION['client_user_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$clientId = $_SESSION['client_id'];
$success = $error = '';

// Get client credentials for Stronpower
$clientStmt = $db->prepare("
    SELECT stron_company, stron_username, stron_password 
    FROM clients 
    WHERE id = ?
");
$clientStmt->execute([$clientId]);
$clientCreds = $clientStmt->fetch();

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'query_credit') {
        require_once '../../includes/StronpowerService.php';
        
        $meterId = $_POST['meter_id'];
        $meterIdValue = $_POST['meter_id_value'];
        
        $credentials = [
            'company_name' => $clientCreds['stron_company'],
            'username' => $clientCreds['stron_username'],
            'password' => $clientCreds['stron_password']
        ];
        
        $stron = new StronpowerService();
        list($ok, $data, $err) = $stron->queryMeterCredit($credentials, $meterIdValue);
        
        if ($ok) {
            // Extract credit from response
            $credit = $data['Credit'] ?? $data['credit'] ?? $data['TotalCredit'] ?? $data['total_credit'] ?? 0;
            
            // Update meter record with last query
            $updateStmt = $db->prepare("
                UPDATE meters 
                SET last_credit_query = NOW(), 
                    last_credit_amount = ?
                WHERE id = ?
            ");
            $updateStmt->execute([$credit, $meterId]);
            
            $success = "Meter Credit: KES " . number_format($credit, 2);
        } else {
            $error = "Failed to query meter credit: " . ($err ?: 'Unknown error');
        }
    } elseif ($action === 'assign') {
        $meterId = $_POST['meter_id'];
        $customerId = $_POST['customer_id'];
        
        // Check if meter is already assigned
        $stmt = $db->prepare("SELECT id FROM meter_assignments WHERE meter_id = ? AND is_active = 1");
        $stmt->execute([$meterId]);
        if ($stmt->fetch()) {
            $error = 'Meter is already assigned to another customer. Unassign first.';
        } else {
            try {
                $db->beginTransaction();
                
                // Create assignment
                $stmt = $db->prepare("
                    INSERT INTO meter_assignments 
                    (meter_id, customer_id, assigned_at, assigned_by, is_active)
                    VALUES (?, ?, NOW(), ?, 1)
                ");
                $stmt->execute([$meterId, $customerId, $_SESSION['client_user_id']]);
                
                // Update meter
                $stmt = $db->prepare("UPDATE meters SET is_assigned = 1 WHERE id = ?");
                $stmt->execute([$meterId]);
                
                $db->commit();
                $success = 'Meter assigned successfully';
            } catch (Exception $e) {
                $db->rollBack();
                $error = 'Error: ' . $e->getMessage();
            }
        }
    } elseif ($action === 'unassign') {
        try {
            $db->beginTransaction();
            
            $assignmentId = $_POST['assignment_id'];
            $meterId = $_POST['meter_id'];
            
            // Deactivate assignment
            $stmt = $db->prepare("
                UPDATE meter_assignments 
                SET is_active = 0, unassigned_at = NOW(), unassigned_by = ?
                WHERE id = ?
            ");
            $stmt->execute([$_SESSION['client_user_id'], $assignmentId]);
            
            // Update meter
            $stmt = $db->prepare("UPDATE meters SET is_assigned = 0 WHERE id = ?");
            $stmt->execute([$meterId]);
            
            $db->commit();
            $success = 'Meter unassigned successfully';
        } catch (Exception $e) {
            $db->rollBack();
            $error = 'Error: ' . $e->getMessage();
        }
    }
}

// Get assignments
$assignments = $db->prepare("
    SELECT ma.*, 
           m.meter_id, m.meter_type, m.location,
           c.customer_id, CONCAT(c.first_name, ' ', c.last_name) as customer_name, c.phone,
           COUNT(t.id) as token_count
    FROM meter_assignments ma
    JOIN meters m ON ma.meter_id = m.id
    JOIN customers c ON ma.customer_id = c.id
    LEFT JOIN tokens t ON m.id = t.meter_id AND t.created_at >= ma.assigned_at
    WHERE m.client_id = ? AND ma.is_active = 1
    GROUP BY ma.id
    ORDER BY ma.assigned_at DESC
");
$assignments->execute([$clientId]);
$assignments = $assignments->fetchAll();

// Get unassigned meters
$unassignedMeters = $db->prepare("
    SELECT id, meter_id, location 
    FROM meters 
    WHERE client_id = ? AND is_active = 1 AND is_assigned = 0
");
$unassignedMeters->execute([$clientId]);
$unassignedMeters = $unassignedMeters->fetchAll();

// Get customers
$customers = $db->prepare("
    SELECT id, customer_id, CONCAT(first_name, ' ', last_name) as name 
    FROM customers 
    WHERE client_id = ? AND is_active = 1
");
$customers->execute([$clientId]);
$customers = $customers->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Meter Assignments</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Meter Assignments</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openModal()" class="btn btn-orange">
                        <i class="fas fa-link"></i> Assign Meter
                    </button>
                    <span><?= htmlspecialchars($_SESSION['client_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>Active Assignments (<?= count($assignments) ?>)</h2>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Meter ID</th>
                                <th>Location</th>
                                <th>Customer</th>
                                <th>Phone</th>
                                <th>Assigned On</th>
                                <th>Tokens</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($assignments as $a): ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($a['meter_id']) ?></strong></td>
                                    <td><?= htmlspecialchars($a['location'] ?: 'N/A') ?></td>
                                    <td>
                                        <?= htmlspecialchars($a['customer_name']) ?><br>
                                        <small><?= htmlspecialchars($a['customer_id']) ?></small>
                                    </td>
                                    <td><?= htmlspecialchars($a['phone']) ?></td>
                                    <td><?= date('Y-m-d H:i', strtotime($a['assigned_at'])) ?></td>
                                    <td><?= $a['token_count'] ?></td>
                                    <td>
                                        <button onclick="queryCredit(<?= $a['meter_id'] ?>, '<?= htmlspecialchars($a['meter_id']) ?>')" 
                                                class="btn btn-sm btn-primary" title="Query Meter Credit">
                                            <i class="fas fa-wallet"></i>
                                        </button>
                                        <form method="POST" style="display:inline">
                                            <input type="hidden" name="action" value="unassign">
                                            <input type="hidden" name="assignment_id" value="<?= $a['id'] ?>">
                                            <input type="hidden" name="meter_id" value="<?= $a['meter_id'] ?>">
                                            <button type="submit" class="btn btn-sm" style="background:#dc3545;color:white" 
                                                    onclick="return confirm('Unassign this meter?')">
                                                <i class="fas fa-unlink"></i> Unassign
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <?php if (count($unassignedMeters) > 0): ?>
                    <div class="mt-4 p-4 bg-white rounded shadow">
                        <h3 class="text-blue mb-2">Unassigned Meters: <?= count($unassignedMeters) ?></h3>
                        <p class="text-sm text-gray">These meters are not assigned to any customer yet.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Assign Modal -->
    <div id="assignModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Assign Meter to Customer</h3>
                <button onclick="closeModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="assign">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Select Meter *</label>
                        <select name="meter_id" required>
                            <option value="">Choose unassigned meter...</option>
                            <?php foreach ($unassignedMeters as $m): ?>
                                <option value="<?= $m['id'] ?>">
                                    <?= htmlspecialchars($m['meter_id']) ?> - <?= htmlspecialchars($m['location'] ?: 'No location') ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Select Customer *</label>
                        <select name="customer_id" required>
                            <option value="">Choose customer...</option>
                            <?php foreach ($customers as $c): ?>
                                <option value="<?= $c['id'] ?>">
                                    <?= htmlspecialchars($c['name']) ?> (<?= htmlspecialchars($c['customer_id']) ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="alert" style="background:#fff3cd;border-left-color:#ffc107;color:#856404">
                        <strong>Note:</strong> One meter can only be assigned to one customer at a time.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Assign Meter</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openModal() {
        document.getElementById('assignModal').classList.add('active');
    }
    function closeModal() {
        document.getElementById('assignModal').classList.remove('active');
    }
    
    function queryCredit(meterId, meterIdValue) {
        if (confirm('Query credit balance for meter ' + meterIdValue + '?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="query_credit">
                <input type="hidden" name="meter_id" value="${meterId}">
                <input type="hidden" name="meter_id_value" value="${meterIdValue}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>