<?php
require_once '../../config/config.php';
require_once '../../includes/DB.php';
require_once '../../includes/VendingService.php';
require_once '../../includes/SMSService.php';

if (!isset($_SESSION['client_user_id'])) {
    header('Location: login.php');
    exit;
}

$db = DB::conn();
$clientId = $_SESSION['client_id'];
$success = $error = '';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'manual_vend') {
        list($ok, $tokenId, $tokenValue) = VendingService::vendManually(
            $_POST['meter_id'],
            $_POST['customer_id'],
            $_POST['amount'],
            isset($_POST['is_vend_by_unit']),
            $_SESSION['client_user_id']
        );
        
        if ($ok) {
            $success = "Token generated: $tokenValue";
        } else {
            $error = $tokenValue ?: 'Vending failed';
        }
    } elseif ($action === 'resend_sms') {
        $tokenId = $_POST['token_id'];
        $phone = $_POST['phone'];
        
        // Get token details
        $stmt = $db->prepare("
            SELECT t.*, m.meter_id 
            FROM tokens t 
            JOIN meters m ON t.meter_id = m.id 
            WHERE t.id = ? AND t.client_id = ?
        ");
        $stmt->execute([$tokenId, $clientId]);
        $token = $stmt->fetch();
        
        if ($token) {
            $sms = new SMSService();
            list($smsOk, $smsResp) = $sms->sendToken(
                $phone,
                $token['meter_id'],
                $token['token_value'],
                $token['amount'],
                $token['units'] ?? 0
            );
            
            if ($smsOk) {
                $stmt = $db->prepare("
                    UPDATE tokens 
                    SET status = 'DELIVERED', delivered_at = NOW(), sms_to = ?, sms_response = ?
                    WHERE id = ?
                ");
                $stmt->execute([$phone, $smsResp, $tokenId]);
                $success = 'SMS sent successfully to ' . $phone;
            } else {
                $error = 'Failed to send SMS';
            }
        } else {
            $error = 'Token not found';
        }
    }
}

// Get tokens
$tokens = $db->prepare("
    SELECT t.*, 
           m.meter_id,
           c.customer_id, CONCAT(c.first_name, ' ', c.last_name) as customer_name, c.phone,
           p.mpesa_tx_id
    FROM tokens t
    JOIN meters m ON t.meter_id = m.id
    JOIN customers c ON t.customer_id = c.id
    LEFT JOIN payments p ON t.payment_id = p.id
    WHERE t.client_id = ?
    ORDER BY t.created_at DESC
    LIMIT 50
");
$tokens->execute([$clientId]);
$tokens = $tokens->fetchAll();

// Get assigned meters with customers
$assignedMeters = $db->prepare("
    SELECT m.id, m.meter_id, c.id as customer_id, 
           CONCAT(c.first_name, ' ', c.last_name) as customer_name, c.phone
    FROM meters m
    JOIN meter_assignments ma ON m.id = ma.meter_id AND ma.is_active = 1
    JOIN customers c ON ma.customer_id = c.id
    WHERE m.client_id = ? AND m.is_active = 1
");
$assignedMeters->execute([$clientId]);
$assignedMeters = $assignedMeters->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Token Management</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="flex">
        <?php include 'sidebar.php'; ?>
        
        <div class="main-content w-full">
            <div class="top-bar">
                <h1 class="text-blue">Token Management</h1>
                <div class="flex items-center gap-4">
                    <button onclick="openVendModal()" class="btn btn-orange">
                        <i class="fas fa-plus"></i> Generate Token
                    </button>
                    <span><?= htmlspecialchars($_SESSION['client_name']) ?></span>
                    <a href="logout.php" class="btn btn-sm" style="background:#dc3545;color:white">Logout</a>
                    <button class="hamburger" onclick="toggleSidebar()"><i class="fas fa-bars"></i></button>
                </div>
            </div>
            
            <div class="content">
                <?php if ($success): ?>
                    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
                <?php endif; ?>
                
                <div class="table-container">
                    <div class="table-header">
                        <h2>Token History (Last 50)</h2>
                        <input type="text" id="search" placeholder="Search..." style="width:250px;padding:0.5rem;border:1px solid #ddd;border-radius:6px">
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Meter</th>
                                <th>Customer</th>
                                <th>Amount</th>
                                <th>Units</th>
                                <th>Token</th>
                                <th>Source</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tokens as $t): ?>
                                <tr>
                                    <td><?= date('Y-m-d H:i', strtotime($t['created_at'])) ?></td>
                                    <td>
                                        <?php
                                        $typeColors = [
                                            'VENDING' => 'badge-success',
                                            'CLEAR_CREDIT' => 'badge-info',
                                            'CLEAR_TAMPER' => 'badge-warning'
                                        ];
                                        ?>
                                        <span class="badge <?= $typeColors[$t['token_type']] ?? 'badge-info' ?>">
                                            <?= str_replace('_', ' ', $t['token_type']) ?>
                                        </span>
                                    </td>
                                    <td><code><?= htmlspecialchars($t['meter_id']) ?></code></td>
                                    <td>
                                        <?= htmlspecialchars($t['customer_name']) ?><br>
                                        <small><?= htmlspecialchars($t['phone']) ?></small>
                                    </td>
                                    <td>KES <?= number_format($t['amount'], 2) ?></td>
                                    <td><?= number_format($t['units'], 2) ?></td>
                                    <td><code style="font-size:0.7rem"><?= htmlspecialchars($t['token_value']) ?></code></td>
                                    <td>
                                        <?php if ($t['is_manual']): ?>
                                            <span class="badge badge-info">Manual</span>
                                        <?php else: ?>
                                            <span class="badge badge-success">Auto</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php
                                        $statusColors = [
                                            'DELIVERED' => 'badge-success',
                                            'CREATED' => 'badge-warning',
                                            'FAILED' => 'badge-danger'
                                        ];
                                        ?>
                                        <span class="badge <?= $statusColors[$t['status']] ?? 'badge-info' ?>">
                                            <?= $t['status'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <button onclick="resendSMS(<?= $t['id'] ?>, '<?= htmlspecialchars($t['phone']) ?>')" 
                                                class="btn btn-sm btn-primary" title="Resend SMS">
                                            <i class="fas fa-sms"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Manual Vend Modal -->
    <div id="vendModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Generate Token Manually</h3>
                <button onclick="closeVendModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="manual_vend">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Select Meter & Customer *</label>
                        <select name="meter_id" id="meterSelect" required onchange="updateCustomer()">
                            <option value="">Choose assigned meter...</option>
                            <?php foreach ($assignedMeters as $m): ?>
                                <option value="<?= $m['id'] ?>" data-customer="<?= $m['customer_id'] ?>" data-phone="<?= htmlspecialchars($m['phone']) ?>">
                                    <?= htmlspecialchars($m['meter_id']) ?> - <?= htmlspecialchars($m['customer_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <input type="hidden" name="customer_id" id="customerId">
                    </div>
                    
                    <div class="form-group">
                        <label>Amount *</label>
                        <input type="number" name="amount" step="0.01" min="1" required placeholder="100.00">
                    </div>
                    
                    <div class="form-group">
                        <label style="display:flex;align-items:center;gap:0.5rem">
                            <input type="checkbox" name="is_vend_by_unit" value="1">
                            <span>Vend by Unit (instead of money)</span>
                        </label>
                    </div>
                    
                    <div class="alert" style="background:#d1ecf1;border-left-color:#17a2b8;color:#0c5460">
                        <strong>Note:</strong> Token will be generated via Stronpower API and SMS will be sent automatically to customer's phone.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeVendModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Generate Token</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Resend SMS Modal -->
    <div id="smsModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Resend Token SMS</h3>
                <button onclick="closeSMSModal()" style="background:none;border:none;color:white;font-size:1.5rem;cursor:pointer">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="resend_sms">
                <input type="hidden" name="token_id" id="smsTokenId">
                <div class="modal-body">
                    <div class="form-group">
                        <label>Send to Phone Number *</label>
                        <input type="text" name="phone" id="smsPhone" placeholder="+254745600377" required>
                        <small style="color:#666">Will be normalized to +254 format</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" onclick="closeSMSModal()" class="btn">Cancel</button>
                    <button type="submit" class="btn btn-orange">Send SMS</button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    function openVendModal() {
        document.getElementById('vendModal').classList.add('active');
    }
    function closeVendModal() {
        document.getElementById('vendModal').classList.remove('active');
    }
    
    function updateCustomer() {
        const select = document.getElementById('meterSelect');
        const option = select.options[select.selectedIndex];
        document.getElementById('customerId').value = option.dataset.customer || '';
    }
    
    function resendSMS(tokenId, phone) {
        document.getElementById('smsTokenId').value = tokenId;
        document.getElementById('smsPhone').value = phone;
        document.getElementById('smsModal').classList.add('active');
    }
    function closeSMSModal() {
        document.getElementById('smsModal').classList.remove('active');
    }
    
    // Search
    document.getElementById('search').addEventListener('input', function(e) {
        const search = e.target.value.toLowerCase();
        document.querySelectorAll('tbody tr').forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(search) ? '' : 'none';
        });
    });
    </script>
    <script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>
</body>
</html>